<?php

namespace App\Http\Controllers;

use App\Models\Schedule;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class ScheduleController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $schedules = Schedule::where('deleted', false)->get();
        return view('schedule.index', compact('schedules'));
        // return response()->json($schedule);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request)
    {
        return view('schedule.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {

            $request['users_id'] =  auth()->id();
            $request['deleted'] =  0;


            $validatedData = $request->validate([
                'title' => 'required|string|max:255',
                'description' => 'required|string',
                'users_id' => 'required|integer|exists:users,id',
                'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg,webp|max:2048', // Imagem opcional
                'deleted' => 'integer',
                'date' => 'date',
                'location' => 'required|max:2000'
            ]);

            // Capturar o ficheiro enviado
            $image = $request->file('image');

            // Obter o nome original do ficheiro
            $originalName = $image->getClientOriginalName();

            // Fazer o upload da imagem para storage/public/photos
            $path = $image->store('eventos', 'public');

            // Atualizar os dados da foto
            $schedule =  Schedule::create([
                'users_id' => auth()->id(),
                'title' => $request->title,
                'description' => $request->description,
                'date' => $request->date,
                'image' => $path, // Usa a nova imagem ou mantém a antiga
                'deleted' => 0,
                'location' => $request['location']
            ]);


            return response()->json([
                'message' => 'Evento inserido com sucesso.',
                'data' => $schedule,
                'response' => 201
            ], 201);
        } catch (QueryException $e) {
            return response()->json(['error' => 'Erro no banco de dados: ' . $e->getMessage()], 500);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json(['error' => 'Erro de validação', 'messages' => $e->errors()], 422);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Erro: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Schedule $schedule)
    {
        try {

            $schedule = Schedule::findOrFail($schedule->id);
            return view('schedule.show', compact('schedule'));

            if (!$schedule) {
                return response()->json(['error' => 'Evento não encontrado.'], 404);
            }

            return response()->json($schedule);
        } catch (QueryException $e) {
            return response()->json(['error' => 'Erro no banco de dados: ' . $e->getMessage()], 500);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Evento não encontrado'], 500);
        }
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Schedule $schedule)
    {
        $schedule = Schedule::findOrFail($schedule->id);
        return view('schedule.edit', compact('schedule'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Schedule $schedule)
    {
        try {
            $request['users_id'] =  auth()->id();
            $request['deleted'] =  0;

            $schedule = Schedule::findOrFail($schedule->id);

            if ($schedule->deleted) {
                return response()->json(['error' => 'Evento não encontrado'], 404);
            }



            $validatedData = $request->validate([
                'title' => 'required|string|max:255',
                'description' => 'required|string',
                'users_id' => 'required|integer|exists:users,id',
                'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg,webp|max:2048', // Imagem opcional
                'deleted' => 'integer',
                'date' => 'date',
                'location' => 'required|max:2000'
            ]);

            // Inicialmente, mantém o caminho da imagem existente
            $path = $schedule->image;
            $originalName = $schedule->original_name;

            // Se houver nova imagem, faz o upload e apaga a imagem antiga
            if ($request->hasFile('image')) {
                $image = $request->file('image');
                $originalName = $image->getClientOriginalName(); // Nome original do ficheiro

                // Fazer upload da nova imagem para storage/public/photos
                $path = $image->store('eventos', 'public');

                // Apagar a imagem antiga, se houver
                if ($schedule->image) {
                    Storage::disk('public')->delete($schedule->image);
                }
            }

            // Atualizar os dados da foto
            $schedule->update([
                'users_id' => auth()->id(),
                'title' => $request->title,
                'description' => $request->description,
                'date' => $request->date,
                'image' => $path, // Usa a nova imagem ou mantém a antiga
                'deleted' => 0,
                'location' => $request['location']
            ]);

            //$schedule->update($validatedData);

            return response()->json([
                'message' => 'Evento atualizado com sucesso.',
                'data' => $schedule,
                'response' => 201
            ]);
        } catch (QueryException $e) {
            return response()->json(['error' => 'Erro no banco de dados: ' . $e->getMessage()], 500);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json(['error' => 'Erro de validação', 'messages' => $e->errors()], 422);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Erro: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Schedule $schedule)
    {
        try {
            $schedule = Schedule::findOrFail($schedule->id);

            $schedule->update(['deleted' => 1]);

            return redirect()->route('schedule.index')->with('success', 'Evento excluído com sucesso');

            return response()->json([
                'message' => 'Evento excluído com sucesso.',
                'data' => $schedule
            ]);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Erro: ' . $e->getMessage()], 500);
        }
    }

    public function getEvents(Schedule $schedule)
    {
        try {
            $events = $schedule->where('deleted', false)
                ->get();

            return response()->json($events);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Nenhum evento foi encontrada.'], 404);
        }
    }


    public function getEvent($id)
    {
        try {
            $event = Schedule::where('deleted', false)->where('id', $id)
                ->get();

            return response()->json($event);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Evento não encontrada.'], 404);
        }
    }
}
